import os
import requests
import json

from dotenv import load_dotenv

load_dotenv(dotenv_path=os.path.join(os.path.dirname(__file__), '..', 'input', 'config.env'))
API_ENDPOINT = os.getenv("API_ENDPOINT")


def query_bo_register(company_id, access_token):
    """Query the BO register using a company ID and save the result to a file."""
    if access_token is None:
        print("❌ No access token provided.")
        return None

    headers = {"Authorization": f"Bearer {access_token}"}
    response = requests.get(API_ENDPOINT.format(company_id=company_id), headers=headers)
    print(response)

    if response.status_code == 200:
        print(f"✅ API response for {company_id}")

        return response.json()

    else:
        print(f"❌ API error for {company_id}: {response.status_code}, {response.text}")
        return None


def flatten_bo_data(data):
    """Flatten the BO register JSON into a list of rows for CSV export."""
    rows = []
    org = data.get("organisasjonsnummer")
    first_reg = data.get("foerstRegistrertIRegisterOverReelleRettighetshavere")

    for reg in data.get("registreringer", []):
        base = {
            "organisasjonsnummer": org,
            "foerstRegistrertIRegisterOverReelleRettighetshavere": first_reg,
            "registreringsid": reg.get("registreringsid"),
            "registreringsstatus": reg.get("registreringsstatus", {}).get("kodenavn"),
            "registrert": reg.get("registrert"),
            "erGjeldendeRegistrering": reg.get("erGjeldendeRegistrering"),
            "reelleRettighetshaverestatus": reg.get("reelleRettighetshaverestatus", {}).get("kodenavn"),
        }

        beneficiaries = reg.get("reelleRettighetshavere", [])
        if beneficiaries:
            for rr in beneficiaries:
                person = rr.get("folkeregistrertPerson", {})
                navn = person.get("navn", {})
                for pos in rr.get("posisjoner", []):
                    row = base.copy()
                    row.update({
                        # Person details
                        "fornavn": navn.get("fornavn"),
                        "mellomnavn": navn.get("mellomnavn"),
                        "etternavn": navn.get("etternavn"),
                        "foedselsdato": person.get("foedselsdato"),
                        "foedselsaar": person.get("foedselsaar"),
                        "erDoed": person.get("erDoed"),
                        "bostedsland": person.get("bostedsland", {}).get("navn"),
                        "statsborgerskap": ",".join([s.get("navn") for s in person.get("statsborgerskap", [])]),
                        # Beneficiary metadata
                        "rr_foerstRegistrert": rr.get("foerstRegistrert"),
                        "rr_endret": rr.get("endret"),
                        "rr_erUnntattFraInnsyn": rr.get("erUnntattFraInnsyn"),
                        # Position details
                        "posisjonstype": pos.get("posisjonstype", {}).get("kodenavn"),
                        "stoerrelsesintervall": pos.get("stoerrelsesintervall", {}).get("kodenavn"),
                        "grunnlagstyper": ",".join([g.get("kodenavn") for g in pos.get("grunnlagstyper", [])]),
                        "mellomliggendeVirksomheter": ",".join(
                            [m.get("norskVirksomhet", {}).get("organisasjonsnummer", "")
                             for m in pos.get("mellomliggendeVirksomheter", [])]
                        )
                    })
                    rows.append(row)
        else:
            row = {**base,
                   "fornavn": None, "mellomnavn": None, "etternavn": None,
                   "foedselsdato": None, "foedselsaar": None, "erDoed": None,
                   "bostedsland": None, "statsborgerskap": None,
                   "rr_foerstRegistrert": None, "rr_endret": None, "rr_erUnntattFraInnsyn": None,
                   "posisjonstype": None, "stoerrelsesintervall": None,
                   "grunnlagstyper": None, "mellomliggendeVirksomheter": None}
            rows.append(row)

    return rows